import type { NextApiRequest, NextApiResponse } from "next";
import { prisma } from "@/lib/prisma";
import { requireAdmin } from "@/lib/admin";
import { z } from "zod";

const UpdatePricingBody = z.object({
  technicianId: z.string().min(1),
  hourlyRate: z.number().min(0).max(1000), // Max €1000/hour seems reasonable
});

export default async function handler(
  req: NextApiRequest,
  res: NextApiResponse
) {
  const session = await requireAdmin(req, res);
  if (!session) return;

  if (req.method === "GET") {
    // Get all technicians with their hourly rates
    try {
      // Get all users and filter for technicians (case-insensitive check)
      const allUsers = await prisma.user.findMany({
        select: {
          id: true,
          name: true,
          email: true,
          hourlyRate: true,
          role: true,
          createdAt: true,
          isExternal: true,
          companyName: true,
          customRole: {
            select: {
              id: true,
              name: true,
              description: true,
            },
          },
          ticketsAssigned: {
            select: {
              normalizedElement: {
                select: {
                  roleElements: {
                    include: {
                      role: {
                        select: {
                          id: true,
                          name: true,
                          description: true,
                        },
                      },
                    },
                  },
                },
              },
            },
          },
        },
        orderBy: { name: "asc" },
      });

      // Filter to only TECHNICIAN role (case-insensitive)
      const technicians = allUsers
        .filter((user) => user.role?.toUpperCase() === "TECHNICIAN")
        .map((technician) => {
          // Extract all unique roles from assigned tickets' elements
          const allRoles = new Map();

          // Add custom role if exists
          if (technician.customRole) {
            allRoles.set(technician.customRole.id, technician.customRole);
          }

          // Add roles from assigned tickets' elements
          technician.ticketsAssigned.forEach((ticket) => {
            if (ticket.normalizedElement?.roleElements) {
              ticket.normalizedElement.roleElements.forEach((roleElement) => {
                allRoles.set(roleElement.role.id, roleElement.role);
              });
            }
          });

          return {
            id: technician.id,
            name: technician.name,
            email: technician.email,
            hourlyRate: technician.hourlyRate,
            role: technician.role,
            createdAt: technician.createdAt,
            isExternal: technician.isExternal,
            companyName: technician.companyName,
            customRole: technician.customRole,
            allRoles: Array.from(allRoles.values()),
          };
        });

      return res.status(200).json({ ok: true, technicians });
    } catch (error) {
      console.error("Error fetching technicians:", error);
      return res
        .status(500)
        .json({ ok: false, error: "Failed to fetch technicians" });
    }
  }

  if (req.method === "PUT") {
    // Update technician hourly rate
    const parsed = UpdatePricingBody.safeParse(req.body);
    if (!parsed.success) {
      return res.status(400).json({
        ok: false,
        error: "Invalid input",
        details: parsed.error.flatten(),
      });
    }

    const { technicianId, hourlyRate } = parsed.data;

    try {
      // Verify the user is a technician
      const technician = await prisma.user.findFirst({
        where: { id: technicianId, role: "TECHNICIAN" },
      });

      if (!technician) {
        return res.status(404).json({
          ok: false,
          error: "Technician not found",
        });
      }

      // Update the hourly rate
      const updated = await prisma.user.update({
        where: { id: technicianId },
        data: { hourlyRate },
        select: {
          id: true,
          name: true,
          email: true,
          hourlyRate: true,
        },
      });

      return res.status(200).json({ ok: true, technician: updated });
    } catch (error) {
      console.error("Error updating hourly rate:", error);
      return res
        .status(500)
        .json({ ok: false, error: "Failed to update hourly rate" });
    }
  }

  return res.status(405).json({ ok: false, error: "Method not allowed" });
}
